// /////////////////////////////////////////////////////////////////////////////
// DR DOBB'S CHALLENGES
//
// Filename       : application.cpp
// Date           : February 2008
//
// Description    : This file defines the Application class creates the main
//                  application.  It is responsible for creating the window,
//                  initializing the core (input, sound, textures, etc),
//                  and releasing the core at the end.  During the main game
//                  loop, it is responsible for updating and rendering the
//                  active GUI.
//
//                  Init() must be called at the beginning of the main program
//                  to all static members from the application core are
//                  initialized.  Release() must also be called at the end of
//                  the main program to release the same static members.
//
// ///////////////////////////////////////////////////////////////////////////
#pragma once



#include <windows.h>
#include <windowsx.h>
#include <d3d9.h>
#include <d3dx9.h>
#include <time.h>
#include <list>

#include "input.h"
#include "sound.h"

#include "game.h"
#include "mainmenu.h"
#include "intro.h"
#include "TextureSection.h"

#pragma comment(lib,"d3d9.lib")
//#pragma comment(lib,"d3dx9d.lib")   // Debug build
#pragma comment(lib,"d3dx9.lib")    // Release build
#pragma comment(lib,"dsound.lib")

#pragma comment(lib,"winmm.lib")


#include "Animation.h"
#include "SpriteBatcher.h"



class GameState;
class Button;

class Application 
{

  protected:

    std::vector<Dobbs::Animation>           m_Tiles;

    std::map<std::string,TextureSection>    m_TextureSections;

    std::map<std::string,LPD3DXFONT>        m_Fonts;

    std::map<std::string,LPDIRECT3DTEXTURE9>  m_mapTextures;

    std::map<std::string,std::list<Sound*> >  m_mapSounds;

    SpriteBatcher           m_TileBatcher;

    bool                    m_BusyLoop;

    HWND                    m_hWnd;

    WINDOWPLACEMENT         m_WPLWindowed;

    D3DPRESENT_PARAMETERS   d3dParam;   // Direct3D parameters
    LPDIRECT3D9             pD3D;       // Direct3D interface
    LPDIRECT3DDEVICE9       m_pD3DDevice;
    ID3DXSprite*            m_pD3DSprite;

    POINT                   m_WindowPosWindowed;

    bool                    isRunning;        // Running flag

    bool                    m_FullScreen;

    bool                    m_MouseCursorVisible;

    int                     m_AutoAnimation;
    int                     m_AutoAnimationDelay;

    GameState*              m_pCurrentState;
    GameState*              m_pNextState;

    std::list<Button*>      m_ListComponents;

    // Timing via QueryPerformanceCounter?
    BOOL                    m_UsingQPF;

    // Timing related
    LONGLONG                m_llQPFTicksPerSec;

    DWORD                   m_LastTicks;
    DWORD                   m_LastTicksPerFrame;

    float                   m_BGOffset;

    bool                    m_ShowCursor;


    float                   GetElapsedTime();

    bool                    LoadTexture( const std::string& TextureName );


  public:

    Input                   m_Input;

    std::map<Dobbs::AnimationIndex,Dobbs::Animation>    m_Animations;



    Application( bool Fullscreen = true );
    ~Application();

    void                    Init();
    void                    Release();   // Release static members

    void                    Update();
    void                    UpdateTimed( const float ElapsedTime );
    void                    Render();    // Render active GUI

    void                    Destroy();   // Destroy window (callback for WindowProc)
    void                    Activate();  // Activate window (callback for WindowProc)

    bool                    IsRunning(); // Check if application is still running

    void                    MouseUpdate( int Buttons, int X, int Y );

    void                    DeleteComponents();

    static LRESULT CALLBACK WindowProc( HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam );

    LRESULT                 WinProc( HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam );

    void                    ChangeGameState( GameState* pState );

    void                    AddGUI( Button* pButton );
    void                    AddFont( const std::string& Name, const std::string& FontName, int size, int weight );

    void                    ToggleWindowMode();
    void                    CloseWindow();

    void                    LoadSettings();
    void                    SaveSettings();

    void                    AddSection( const std::string& Section, const std::string& TextureFile, int X, int Y, int W, int H );
    TextureSection          Section( const std::string& Section );

    DWORD                   Run();

    void                    AddTile( Dobbs::TileType Type, const std::string& TexSection );
    TextureSection          Tile( size_t TileIndex, size_t AnimFrame = 0 );

    void                    AddFrameToAnimation( Dobbs::AnimationIndex AnimIndex, const std::string& TexSection, float FrameTime = 1000.0f );
    void                    AddFramesToAnimation( Dobbs::AnimationIndex AnimIndex, const std::string& TexSection, int StartIndex, int EndIndex, float FrameTime = 1000.0f );

    void                    RenderTextureSection( const TextureSection& Section, int X, int Y, DWORD Color = 0xffffffff, float Z = 0.99f );
    void                    RenderTextureSectionRotated( const TextureSection& TexSec, int X, int Y, float Angle, DWORD Color = 0xffffffff );
    void                    RenderTextureSection( const std::string& Section, int X, int Y, DWORD Color = 0xffffffff, float Z = 0.99f );
    void                    RenderTextureSectionRotated( const std::string& TexSec, int X, int Y, float Angle, DWORD Color = 0xffffffff );

    void                    RenderTextureSection( const TextureSection& TexSec, int X1, int Y1, int X2, int Y2, int X3, int Y3, int X4, int Y4, DWORD Color, float Z = 0.99f );
    void                    RenderTextureSectionWhitened( const TextureSection& TexSec, int X, int Y, DWORD Color, float Z = 0.99f );

    void                    RenderTile( Dobbs::TileType Tile, int X, int Y, DWORD Color = 0xffffffff );
    void                    RenderTile( Dobbs::TileType Tile, int X1, int Y1, int X2, int Y2, int X3, int Y3, int X4, int Y4, DWORD Color );

    void                    RenderBatchedTiles();

    void                    RenderText( const std::string& Font, const std::string& Text, int X, int Y, DWORD Color = 0xffffffff );
    void                    RenderTextCentered( const std::string& Font, const std::string& Text, int X, int Y, DWORD Color = 0xffffffff );
    void                    RenderTextRightAligned( const std::string& Font, const std::string& Text, int X, int Y, DWORD Color = 0xffffffff );
    void                    RenderTextInArea( const std::string& Font, const std::string& Text, RECT& Area, DWORD Color = 0xffffffff );

    void                    RenderBackground();
    void                    ForceSpriteDraw();

    void                    Clear();

    void                    LoadSound( const std::string& SoundName, const std::string& Filename );
    void                    PlaySound( const std::string& Sound );
    void                    LoopSound( const std::string& SoundName );
    void                    StopSound( const std::string& SoundName );
    bool                    IsSoundPlaying( const std::string& SoundName );

    void                    EnableCursor( bool Show = true );

};



extern Application        g_App;